# filepath: c:\Users\dpascoal\Documents\Projects\DHLAllinone\src\scripts\check-all-outlook-emails-for-tools.ps1
param(
    [Parameter(Mandatory=$true)][string]$ToolDataJson,
    [Parameter(Mandatory=$false)][string]$OutputJsonPath = (Join-Path $env:TEMP "email_results.json"), # Default to temp directory
    [Parameter(Mandatory=$false)][string]$AppLogFilePath # Optional: For consistency if needed later
)

# --- Configuration ---
$SharedMailboxName = "DHL Service Tools"
$TargetEmailAddress = "asml_tc-ts-suppliermanagement@asml.com" # Lowercase for comparison
$TargetDisplayName = "ASML_TC-TS Supplier Management"
$MaxRetryAttempts = 3
$SearchLookbackDays = 90 # Optional: Limit initial email scan to recent emails for performance
# --- End Configuration ---

# --- Helper Functions ---
function Write-DebugLog {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logEntry = "$timestamp - DEBUG [ALL-EMAILS]: $Message"
    Write-Host $logEntry # Output for Electron process capture

    if (![string]::IsNullOrEmpty($AppLogFilePath)) {
        try {
            Add-Content -Path $AppLogFilePath -Value "[$timestamp] [DEBUG] [ALL-OUTLOOK-EMAILS] $Message" -Encoding UTF8 -ErrorAction SilentlyContinue
        } catch {
            # Silently ignore log file write errors for this script
        }
    }
}

function Write-ScriptResult {
    param(
        [Parameter(Mandatory=$true)][bool]$Success,
        [Parameter(Mandatory=$false)][string]$Message
    )
    Write-Host "SCRIPT_RESULT_START"
    Write-Host "Success=$Success"
    if ($Message) {
        Write-Host "Message=$Message"
    }
    Write-Host "SCRIPT_RESULT_END"
}

# --- Main Script Logic ---
$Outlook = $null
$Namespace = $null
$SharedInbox = $null
$SharedSentItems = $null
$allRelevantEmails = New-Object System.Collections.Generic.List[System.Object] # Store raw matches
$finalResults = @{} # Hashtable to store results grouped by Tool ID (SN/EQ)

try {
    Write-DebugLog "--- Starting Bulk Email Check ---"
    Write-DebugLog "Output JSON Path: $OutputJsonPath"

    # 1. Parse Input Tool Data
    Write-DebugLog "Parsing ToolDataJson..."
    $tools = $null
    try {
        $tools = $ToolDataJson | ConvertFrom-Json -ErrorAction Stop
        if ($null -eq $tools -or $tools.Count -eq 0) {
            throw "Input ToolDataJson was empty or invalid."
        }
        # Create lookup tables for faster checking
        $toolLookup = @{}
        $earliestBookedDate = [DateTime]::MaxValue
        foreach ($tool in $tools) {
            $bookedDate = $null
            if (![string]::IsNullOrWhiteSpace($tool.ChangedDate) -and $tool.ChangedDate -match '^\d{8}$') {
                try {
                    $bookedDate = [DateTime]::ParseExact($tool.ChangedDate, 'yyyyMMdd', $null)
                    if ($bookedDate -lt $earliestBookedDate) {
                        $earliestBookedDate = $bookedDate
                    }
                } catch {
                    Write-DebugLog "Warning: Could not parse ChangedDate '$($tool.ChangedDate)' for tool SN:$($tool.SerialNumber)/EQ:$($tool.EquipmentNumber). Skipping date filter for this tool."
                    $bookedDate = $null # Ensure it's null if parse fails
                }
            } else {
                 Write-DebugLog "Warning: Invalid or missing ChangedDate '$($tool.ChangedDate)' for tool SN:$($tool.SerialNumber)/EQ:$($tool.EquipmentNumber). Skipping date filter for this tool."
                 $bookedDate = $null
            }

            $toolEntry = @{ BookedDate = $bookedDate; SerialNumber = $tool.SerialNumber; EquipmentNumber = $tool.EquipmentNumber }

            if (![string]::IsNullOrWhiteSpace($tool.SerialNumber)) {
                $toolLookup[$tool.SerialNumber.ToLower()] = $toolEntry # Use lowercase keys
            }
            if (![string]::IsNullOrWhiteSpace($tool.EquipmentNumber)) {
                $toolLookup[$tool.EquipmentNumber.ToLower()] = $toolEntry # Use lowercase keys
            }
        }
        Write-DebugLog "Parsed $($tools.Count) tools. Found $($toolLookup.Count) unique SN/EQ identifiers."
        Write-DebugLog "Earliest BookedDate found: $($earliestBookedDate.ToString('yyyy-MM-dd'))"

    } catch {
        Write-DebugLog "FATAL: Failed to parse ToolDataJson: $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Invalid input tool data."
        exit 1 # Use non-zero exit code for fatal input error
    }

    # 2. Connect to Outlook
    Write-DebugLog "Connecting to Outlook..."
    try {
        Add-Type -assembly "Microsoft.Office.Interop.Outlook" -ErrorAction Stop
    } catch {
        Write-DebugLog "FATAL: Failed to load Outlook Interop Assembly: $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Outlook Interop Assembly not found."
        exit 1
    }

    $retryCount = 0
    $outlookConnected = $false
    while (-not $outlookConnected -and $retryCount -lt $MaxRetryAttempts) {
        try {
            $Outlook = [System.Runtime.InteropServices.Marshal]::GetActiveObject('Outlook.Application')
            $outlookConnected = $true
            Write-DebugLog "Connected to running Outlook instance."
        } catch {
            $retryCount++
            Write-DebugLog "Failed to get active Outlook (Attempt $retryCount/$MaxRetryAttempts): $($_.Exception.Message)"
            if ($retryCount -ge $MaxRetryAttempts) {
                try {
                    Write-DebugLog "Trying to start new Outlook instance..."
                    $Outlook = New-Object -ComObject Outlook.Application
                    Start-Sleep -Seconds 5
                    $outlookConnected = $true
                    Write-DebugLog "New Outlook instance started."
                } catch {
                     Write-DebugLog "FATAL: Failed to start or connect to Outlook: $($_.Exception.Message)"
                     Write-ScriptResult -Success $false -Message "Could not connect to Outlook."
                     exit 1
                }
            } else {
                Start-Sleep -Seconds 2
            }
        }
    }

    $Namespace = $Outlook.GetNamespace("MAPI")

    # 3. Access Shared Mailbox Folders
    Write-DebugLog "Accessing shared mailbox '$SharedMailboxName'..."
    $mailboxAccessSuccess = $false
    $retryCount = 0
    while (-not $mailboxAccessSuccess -and $retryCount -lt $MaxRetryAttempts) {
         try {
            $currentMailboxFolder = $null
            $Namespace.Folders | ForEach-Object {
                if ($_.Name -eq $SharedMailboxName) {
                    $currentMailboxFolder = $_
                    Write-DebugLog "Found shared mailbox by name: $SharedMailboxName"
                }
            }

            if ($null -eq $currentMailboxFolder) {
                 $availableMailboxes = $Namespace.Folders | ForEach-Object { $_.Name }
                 Write-DebugLog "Shared mailbox '$SharedMailboxName' not found directly. Available: $($availableMailboxes -join ', ')"
                 # Attempt to get recipient object and then the folder
                 $recipient = $Namespace.CreateRecipient($SharedMailboxName)
                 $recipient.Resolve()
                 if ($recipient.Resolved) {
                     Write-DebugLog "Recipient '$SharedMailboxName' resolved. Attempting GetSharedDefaultFolder."
                     # Try getting Inbox first
                     try {
                         $SharedInbox = $Namespace.GetSharedDefaultFolder($recipient, [Microsoft.Office.Interop.Outlook.OlDefaultFolders]::olFolderInbox)
                         $currentMailboxFolder = $SharedInbox.Parent # Get the root folder of the shared mailbox
                         Write-DebugLog "Accessed shared mailbox '$($currentMailboxFolder.Name)' via resolved recipient (Inbox)."
                     } catch {
                         Write-DebugLog "Could not get shared Inbox via recipient. Trying Sent Items..."
                         # Try getting Sent Items if Inbox failed
                         try {
                            $SharedSentItems = $Namespace.GetSharedDefaultFolder($recipient, [Microsoft.Office.Interop.Outlook.OlDefaultFolders]::olFolderSentMail)
                            $currentMailboxFolder = $SharedSentItems.Parent
                            Write-DebugLog "Accessed shared mailbox '$($currentMailboxFolder.Name)' via resolved recipient (Sent Items)."
                         } catch {
                             Write-DebugLog "Could not get shared Sent Items via recipient either."
                             throw "Could not access shared mailbox '$SharedMailboxName' via recipient resolution."
                         }
                     }
                 } else {
                     throw "Could not resolve recipient '$SharedMailboxName'."
                 }
            }

            # Get Inbox and Sent Items from the found mailbox folder
            if ($null -eq $SharedInbox) { # If not already set via GetSharedDefaultFolder
                $SharedInbox = $currentMailboxFolder.Folders | Where-Object { $_.Name -match 'Inbox|Posteingang' } | Select-Object -First 1
                if ($null -eq $SharedInbox) { $SharedInbox = $currentMailboxFolder.Folders.Item("Inbox") } # Fallback
            }
            if ($null -eq $SharedSentItems) { # If not already set via GetSharedDefaultFolder
                $SharedSentItems = $currentMailboxFolder.Folders | Where-Object { $_.Name -match 'Sent Items|Gesendete Elemente' } | Select-Object -First 1
                if ($null -eq $SharedSentItems) { $SharedSentItems = $currentMailboxFolder.Folders.Item("Sent Items") } # Fallback
            }

            if ($null -eq $SharedInbox) { throw "Could not find Inbox folder in mailbox '$($currentMailboxFolder.Name)'." }
            if ($null -eq $SharedSentItems) { throw "Could not find Sent Items folder in mailbox '$($currentMailboxFolder.Name)'." }

            Write-DebugLog "Using mailbox: $($currentMailboxFolder.Name)"
            Write-DebugLog "Using Inbox folder: $($SharedInbox.Name)"
            Write-DebugLog "Using Sent Items folder: $($SharedSentItems.Name)"
            $mailboxAccessSuccess = $true

         } catch {
            $retryCount++
            Write-DebugLog "Error accessing mailbox (Attempt $retryCount/$MaxRetryAttempts): $($_.Exception.Message)"
            if ($retryCount -ge $MaxRetryAttempts) {
                Write-DebugLog "FATAL: Error accessing mailbox '$SharedMailboxName': $($_.Exception.Message)"
                Write-ScriptResult -Success $false -Message "Error accessing mailbox '$SharedMailboxName'."
                exit 1
            }
            Start-Sleep -Seconds 3
         }
    }

    # 4. Scan Emails and Collect Raw Matches
    Write-DebugLog "Scanning emails (lookback approx $($SearchLookbackDays) days)..."
    $processedCount = 0
    $rawMatchCount = 0
    $scanStartTime = Get-Date

    # Determine the date for the Outlook restriction filter
    $filterDateForOutlook = $null
    if ($earliestBookedDate -ne [DateTime]::MaxValue) {
        # If we have a valid earliest booked date from tools, use it
        $filterDateForOutlook = $earliestBookedDate
        Write-DebugLog "Using earliest BookedDate from tools for initial Outlook filter: $($filterDateForOutlook.ToString('yyyy-MM-dd'))"
    } else {
        # Fallback to SearchLookbackDays if no booked dates were found in tools
        $filterDateForOutlook = (Get-Date).AddDays(-$SearchLookbackDays)
        Write-DebugLog "No valid BookedDates found in tools. Using general lookback of $SearchLookbackDays days for initial Outlook filter: $($filterDateForOutlook.ToString('yyyy-MM-dd'))"
    }
    # Ensure MM/dd/yyyy format is correctly interpreted by Outlook, using en-US culture
    $restrictionDateString = $filterDateForOutlook.ToString("MM/dd/yyyy hh:mm tt", [System.Globalization.CultureInfo]::GetCultureInfo('en-US'))
    $filter = "([ReceivedTime] >= '$restrictionDateString' OR [SentOn] >= '$restrictionDateString')"
    Write-DebugLog "Applying Outlook filter: $filter"

    # Process Inbox
    $inboxItems = $null
    try {
        Write-DebugLog "Processing Inbox '$($SharedInbox.Name)'..."
        $inboxItems = $SharedInbox.Items.Restrict($filter)
        $inboxItems.Sort("[ReceivedTime]", $true) # Sort descending (newest first)
        Write-DebugLog "Found approx $($inboxItems.Count) items in Inbox matching date filter."

        foreach ($item in $inboxItems) {
            $processedCount++
            $itemSubject = "N/A"
            try {
                $itemSubject = $item.Subject
                $itemBody = $item.Body
                $itemSenderAddress = $item.SenderEmailAddress.ToLower() # Use address for matching
                $itemSenderName = $item.SenderName
                $itemRecipients = ""
                foreach ($recipient in $item.Recipients) {
                    try {
                        $itemRecipients += $recipient.Address.ToLower() + ";" # Use address for matching
                    } catch {
                         # Ignore errors getting individual recipient addresses
                    }
                }
                $itemReceivedTime = $item.ReceivedTime
                $itemSentOn = $item.SentOn # Keep SentOn in case ReceivedTime is null
                $itemEntryID = $item.EntryID
                $itemConversationID = $item.ConversationID

                # Check 1: Does it involve the target email/display name?
                $involvesTarget = $false
                $reasonIncluded = ""
                
                # First check: Does the email come from or go to the ASML target?
                if ($itemSenderAddress -like "*$TargetEmailAddress*") {
                    $involvesTarget = $true
                    $reasonIncluded = "Sender address matches ASML target"
                }
                elseif ($itemRecipients -like "*$TargetEmailAddress*") {
                    $involvesTarget = $true
                    $reasonIncluded = "Recipients include ASML target"
                }
                elseif ($itemSenderName -ilike "*$TargetDisplayName*") {
                    $involvesTarget = $true
                    $reasonIncluded = "Sender name matches ASML target"
                }
                
                # Second check: Also include emails with our shared mailbox in recipients
                # This is CRITICAL for catching outgoing/original emails
                if (-not $involvesTarget) {
                    # First check by exact email match if possible
                    if ($itemRecipients -like "*dhl.service.tools@asml.com*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox email found in recipients"
                    }
                    # Then try by display name
                    elseif ($itemRecipients -like "*$SharedMailboxName*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox display name found in recipients"
                    }
                    # Add checks for internal Exchange addressing with the DHL Service Tools name
                    elseif ($itemRecipients -like "*dhl service*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox name pattern found in recipients"
                    }
                    elseif ($itemRecipients -like "*75fa2d0a9451469b8012d203ba33c5a3-dhl service*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox Exchange ID found in recipients"
                    }
                    # Also include emails sent to this mailbox (found in the To/CC field)
                    elseif ($itemSenderAddress -like "*dhl.service.tools@asml.com*" -or $itemSenderName -like "*DHL Service Tools*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Sent from shared mailbox"
                    }
                    # Fallback - any email containing our tools is relevant
                    elseif ($containsIdentifiers) {
                        $involvesTarget = $true
                        $reasonIncluded = "Contains tool identifiers and related to DHL service"
                    }
                }

                # IMPORTANT DEBUG: Log all emails that have tool identifiers, whether included or not
                $containsIdentifiers = $false
                $foundIdentifiersForDebug = New-Object System.Collections.Generic.List[string]
                foreach ($key in $toolLookup.Keys) {
                    $pattern = [regex]::Escape($key)
                    if (($itemSubject -match $pattern) -or ($itemBody -match $pattern)) {
                        $containsIdentifiers = $true
                        $foundIdentifiersForDebug.Add($key)
                    }
                }
                
                if ($containsIdentifiers) {
                    # Only debug emails that contain tool identifiers to minimize noise
                    if ($involvesTarget) {
                        Write-DebugLog "MATCH: Email subject '$itemSubject' included: $reasonIncluded"
                    } else {
                        # This is critical debugging for emails that aren't being included
                        Write-DebugLog "NOT MATCHED: Email subject '$itemSubject' contains tools ${foundIdentifiersForDebug} but was NOT included"
                        Write-DebugLog "   Sender: '$itemSenderName' <$itemSenderAddress>"
                        Write-DebugLog "   Recipients: $itemRecipients"
                    }
                }

                if ($involvesTarget) {
                    # Check 1: Does it contain any SN/EQ from our list?
                    $foundIdentifiers = New-Object System.Collections.Generic.List[string]
                    $foundIdentifiersForDebug = New-Object System.Collections.Generic.List[string]
                    $containsIdentifiers = $false
                    
                    foreach ($key in $toolLookup.Keys) {
                        # Use regex matching for robustness (escape key)
                        $pattern = [regex]::Escape($key)
                        if (($itemSubject -match $pattern) -or ($itemBody -match $pattern)) {
                            $foundIdentifiers.Add($key) # Add the matched key (lowercase SN/EQ)
                            $foundIdentifiersForDebug.Add($key)
                            $containsIdentifiers = $true
                        }
                    }
                    
                    # STRICT EMAIL MATCHING LOGIC PER CLIENT REQUIREMENTS:
                    $emailMatches = $false
                    $reasonIncluded = ""
                    
                    # Get raw To and CC fields as they are most reliable
                    $rawTo = ""
                    $rawCC = ""
                    try { $rawTo = $item.To } catch {}
                    try { $rawCC = $item.CC } catch {}
                    
                    # Convert everything to lowercase for case-insensitive comparison
                    $toLower = $rawTo.ToLower()
                    $ccLower = $rawCC.ToLower() 
                    $senderAddressLower = $itemSenderAddress.ToLower()
                    $targetEmailLower = "asml_tc-ts-suppliermanagement@asml.com"
                    $sharedMailboxLower = "dhl.service.tools@asml.com"
                    
                    # CASE 1: INCOMING EMAIL (A REPLY) - FROM THE TARGET EMAIL AND SHARED MAILBOX IN TO OR CC
                    if ($senderAddressLower -like "*$targetEmailLower*" -and 
                        ($toLower -like "*$sharedMailboxLower*" -or $ccLower -like "*$sharedMailboxLower*")) {
                        $emailMatches = $true
                        $reasonIncluded = "Received from ASML SupplierManagement with our mailbox in TO/CC"
                    }
                    
                    # CASE 2: OUTGOING EMAIL - TO TARGET EMAIL AND CC'D TO SHARED MAILBOX
                    elseif (($toLower -like "*$targetEmailLower*") -and ($ccLower -like "*$sharedMailboxLower*")) {
                        $emailMatches = $true
                        $reasonIncluded = "Sent to ASML SupplierManagement with shared mailbox CC'd"
                    }
                    
                    # Debug logging for emails that contain tool identifiers
                    if ($containsIdentifiers) {
                        if ($emailMatches) {
                            Write-DebugLog "MATCH: Email subject '$itemSubject' included: $reasonIncluded"
                        } else {
                            Write-DebugLog "NOT MATCHED: Email subject '$itemSubject' contains tools $($foundIdentifiersForDebug -join ' ') but was NOT included"
                            Write-DebugLog "   Sender: '$itemSenderName' <$senderAddressLower>"
                            Write-DebugLog "   To: $toLower"
                            Write-DebugLog "   CC: $ccLower"
                        }
                    }

                    # Add if it matches identifiers (NO check for IsReply needed here)
                    if ($foundIdentifiers.Count -gt 0) {
                        $rawMatchCount++
                        # Store relevant data
                        $recipientNames = ($item.Recipients | ForEach-Object { $_.Name }) -join '; '
                        $emailData = [PSCustomObject]@{
                            Subject = $itemSubject
                            Body = $itemBody
                            Sender = $itemSenderName
                            Recipients = $recipientNames
                            Date = if ($itemReceivedTime) { $itemReceivedTime.ToString("yyyy-MM-dd HH:mm") } elseif ($itemSentOn) { $itemSentOn.ToString("yyyy-MM-dd HH:mm") } else { "Unknown Date" }
                            DateTimeObj = if ($itemReceivedTime) { $itemReceivedTime } elseif ($itemSentOn) { $itemSentOn } else { $null }
                            IsReply = ($itemSubject -match '^(RE|AW|WG|FW|VS|VB):') # Still determine if it IS a reply
                            IsSentItem = $false # Always false now
                            FolderName = $SharedInbox.Name
                            EntryID = $itemEntryID
                            ConversationID = $itemConversationID
                            MatchedIDs = $foundIdentifiers
                        }
                        $allRelevantEmails.Add($emailData)
                    }
                }

            } catch {
                Write-DebugLog "Error processing item #$processedCount (Subject: '$itemSubject'): $($_.Exception.Message)"
            } finally {
                if ($null -ne $item) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($item) | Out-Null } catch {} }
            }
        }
        Write-DebugLog "Finished Inbox. Processed $processedCount items total. Found $rawMatchCount raw matches (from Inbox only)." # Updated log message

    } catch {
         Write-DebugLog "ERROR processing Inbox items: $($_.Exception.Message)"
    } finally {
         if ($null -ne $inboxItems) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($inboxItems) | Out-Null } catch {} }
    }

    # --- SENT ITEMS PROCESSING BLOCK REMAINS REMOVED ---

    $scanEndTime = Get-Date
    Write-DebugLog "Email scanning finished in $(($scanEndTime - $scanStartTime).TotalSeconds) seconds."

    # 5. Filter Raw Matches by BookedDate and Group by Tool ID
    Write-DebugLog "Filtering $rawMatchCount raw matches against tool BookedDates..."
    $finalMatchCount = 0

    # Initialize finalResults with all tool keys from lookup
    foreach ($key in $toolLookup.Keys) {
        $finalResults[$key] = New-Object System.Collections.Generic.List[System.Object]
    }

    foreach ($email in $allRelevantEmails) {
        if ($null -eq $email.DateTimeObj) {
            Write-DebugLog "Skipping email (Subject: '$($email.Subject)') due to missing DateTimeObj."
            continue # Skip if date couldn't be parsed/stored
        }

        foreach ($matchedId in $email.MatchedIDs) { # Iterate through SN/EQ keys found in this email
            if ($toolLookup.ContainsKey($matchedId)) {
                $toolInfo = $toolLookup[$matchedId]
                # Check if email date is >= tool's booked date (if booked date is valid)
                if ($null -eq $toolInfo.BookedDate -or $email.DateTimeObj -ge $toolInfo.BookedDate) {
                    # Add a *copy* of the email data, removing MatchedIDs and DateTimeObj
                    $outputEmailData = $email | Select-Object * -ExcludeProperty MatchedIDs, DateTimeObj
                    $finalResults[$matchedId].Add($outputEmailData)
                    $finalMatchCount++
                    # Write-DebugLog "Final Match: Email '$($email.Subject)' added to tool '$matchedId' (Date $($email.Date) >= BookedDate $($toolInfo.BookedDate))" # Too verbose
                } else {
                    # Write-DebugLog "Filtered out: Email '$($email.Subject)' for tool '$matchedId' (Date $($email.Date) < BookedDate $($toolInfo.BookedDate))" # Too verbose
                }
            } else {
                 Write-DebugLog "Warning: Matched ID '$matchedId' from email not found in tool lookup (should not happen)."
            }
        }
    }
    Write-DebugLog "Filtering complete. Found $finalMatchCount relevant email instances across all tools."

    # 6. Convert Final Results to JSON and Save
    Write-DebugLog "Converting final results to JSON..."
    $jsonOutput = $null
    try {
        # Convert the hashtable $finalResults to JSON
        # Need to handle the List[Object] inside the hashtable correctly
        $jsonOutput = $finalResults | ConvertTo-Json -Depth 5 -Compress -ErrorAction Stop
        Write-DebugLog "JSON conversion successful."
    } catch {
        Write-DebugLog "FATAL: Failed to convert results to JSON: $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Failed to generate JSON output."
        exit 1
    }

    Write-DebugLog "Saving JSON to '$OutputJsonPath'..."
    try {
        # Ensure directory exists
        $outputDir = Split-Path -Path $OutputJsonPath -Parent
        if (-not (Test-Path -Path $outputDir)) {
            New-Item -ItemType Directory -Path $outputDir -Force | Out-Null
            Write-DebugLog "Created output directory: $outputDir"
        }
        # Write JSON content WITHOUT BOM
        # Use [System.IO.File]::WriteAllText for better control over encoding/BOM
        $utf8EncodingWithoutBom = New-Object System.Text.UTF8Encoding($false) # $false specifies no BOM
        [System.IO.File]::WriteAllText($OutputJsonPath, $jsonOutput, $utf8EncodingWithoutBom)
        # Set-Content -Path $OutputJsonPath -Value $jsonOutput -Encoding UTF8 -ErrorAction Stop # <-- Replaced this line
        Write-DebugLog "Successfully saved JSON output."
        Write-ScriptResult -Success $true -Message "Email check complete. Results saved to $OutputJsonPath"

    } catch {
        Write-DebugLog "FATAL: Failed to write JSON to '$OutputJsonPath': $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Failed to save JSON output file."
        exit 1
    }

} catch {
    # Catch any unexpected top-level errors
    Write-DebugLog "FATAL SCRIPT ERROR: $($_.Exception.ToString())"
    Write-ScriptResult -Success $false -Message "Bulk email check encountered a fatal script error."
    exit 1 # Use non-zero exit code for fatal script errors
} finally {
    # --- COM Object Cleanup ---
    Write-DebugLog "Cleaning up COM objects..."
    if ($null -ne $SharedSentItems) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($SharedSentItems) | Out-Null } catch {} }
    if ($null -ne $SharedInbox) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($SharedInbox) | Out-Null } catch {} }
    if ($null -ne $currentMailboxFolder) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($currentMailboxFolder) | Out-Null } catch {} }
    if ($null -ne $Namespace) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($Namespace) | Out-Null } catch {} }
    if ($null -ne $Outlook) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($Outlook) | Out-Null } catch {} }
    [System.GC]::Collect()
    [System.GC]::WaitForPendingFinalizers()
    Write-DebugLog "--- Bulk Email Check Script Finished ---"
}
